# coding: utf-8

import os
import sys
import subprocess
import urllib.request
import zipfile
import logging
import re
from io import BytesIO
import platform as plat
def get_edge_version():
    """
    Detects the installed version of Microsoft Edge.
    
    :return: Edge version string or None if not found
    """
    platform, _ = get_platform_architecture()

    try:
        if platform == 'linux':
            executable_name = 'microsoft-edge'
            if os.path.isfile('/usr/bin/microsoft-edge-stable'):
                executable_name = 'microsoft-edge-stable'
            if os.path.isfile('/usr/bin/microsoft-edge-dev'):
                executable_name = 'microsoft-edge-dev'
            with subprocess.Popen([executable_name, '--version'], stdout=subprocess.PIPE) as proc:
                version = proc.stdout.read().decode('utf-8').replace('Microsoft Edge', '').strip()

        elif platform == 'mac':
            process = subprocess.Popen(
                ['/Applications/Microsoft Edge.app/Contents/MacOS/Microsoft Edge', '--version'],
                stdout=subprocess.PIPE
            )
            version = process.communicate()[0].decode('utf-8').replace('Microsoft Edge', '').strip()

        elif platform == 'win':
            process = subprocess.Popen(
                ['reg', 'query', 'HKEY_CURRENT_USER\\Software\\Microsoft\\Edge\\BLBeacon', '/v', 'version'],
                stdout=subprocess.PIPE, stderr=subprocess.DEVNULL, stdin=subprocess.DEVNULL
            )
            version = process.communicate()[0].decode('utf-8').strip().split()[-1]

        else:
            return None

        return version

    except Exception as e:
        logging.error("No se pudo obtener la versión de Edge: %s", e)
        return None


def get_platform_architecture():
    """
    Returns the platform and architecture used to build the EdgeDriver download URL.
    """
    if sys.platform.startswith('linux') and sys.maxsize > 2 ** 32:
        platform = 'linux'
        architecture = '64'
    elif sys.platform == 'darwin':
        if plat.machine() == 'x86_64':
            platform = 'mac'
            architecture = '64'
        elif plat.machine() == 'arm64':
            platform = 'mac'
            architecture = '64_m1'
    elif sys.platform.startswith('win'):
        platform = 'win'
        architecture = '32' if sys.maxsize <= 2**32 else '64'
    else:
        raise RuntimeError('Could not determine EdgeDriver platform.')
    return platform, architecture

def get_variable_separator():
    return ';' if os.name == 'nt' else ':'

def get_edgedriver_url(version):
    """
    Generates the download URL for current platform, architecture and the given version.
    Supports Linux, MacOS and Windows.
    :param version: edgedriver version string
    :return: Download URL for edgedriver
    """
    platform, architecture = get_platform_architecture()
    zip_name = f"edgedriver_{platform}{architecture}.zip"
    return f"https://msedgedriver.microsoft.com/{version}/{zip_name}"


def check_version(binary, required_version):
    try:
        version = subprocess.check_output([binary, '--version'])
        version = re.match(r'.*?([\d.]+).*?', version.decode('utf-8'))[1]
        return version == required_version
    except Exception:
        return False

def is_updated(edgedriver_dir):
    edge_version = get_edge_version()
    if not edge_version:
        logging.debug("Edge is not installed or version is unknown..")
        return True  # No intenta actualizar si no se detecta

    edgedriver_path = os.path.join(edgedriver_dir, 'msedgedriver.exe')
    return check_version(edgedriver_path, edge_version)

def download_edgedriver(edgedriver_dir, cwd=False):
    """
    Downloads, unzips, and installs EdgeDriver for the detected Microsoft Edge version.
    Handles subfolders inside the zip, removes previous binaries, and applies correct permissions.
    
    :param edgedriver_dir: Destination directory
    :param cwd: (Unused for now)
    :return: Full path to installed edgedriver or None if failed
    """
    edge_version = get_edge_version()
    if not edge_version:
        logging.debug('Edge is not installed or version is unknown.')
        return None

    edgedriver_filename = 'msedgedriver.exe'
    last_path = os.path.join(edgedriver_dir, edgedriver_filename)

    if not os.path.isdir(edgedriver_dir):
        os.makedirs(edgedriver_dir)

    needs_download = (
        not os.path.isfile(last_path) or
        not check_version(last_path, edge_version)
    )

    if needs_download:
        print(f"Downloading EdgeDriver ({edge_version})...")
        logging.debug(f"Downloading EdgeDriver version: {edge_version}")

        url = get_edgedriver_url(edge_version)

        try:
            response = urllib.request.urlopen(url, timeout=30)
            if response.status != 200:
                raise urllib.error.URLError(f"Status code: {response.status}")
        except Exception as e:
          # fallback with other url
            try:
                url = url.replace('msedgedriver.microsoft.com', 'msedgedriver.azureedge.net')
                response = urllib.request.urlopen(url, timeout=30)
                if response.status != 200:
                    raise urllib.error.URLError(f"Status code: {response.status}")
            except Exception as e:
                print(f"Failed to download EdgeDriver: {e}")
                logging.error(f"Download error for EdgeDriver: {e}")
                return None

        # Process ZIP archive
        archive = BytesIO(response.read())
        with zipfile.ZipFile(archive) as zip_file:
            # Extract only the driver binary
            for file in zip_file.namelist():
                if file.endswith(edgedriver_filename):
                    extracted_path = zip_file.extract(file, edgedriver_dir)
                    break
            else:
                print("EdgeDriver executable not found in the archive.")
                logging.error("EdgeDriver executable not found in the zip file.")
                return None

        # Remove existing binary if needed
        # if os.path.isfile(last_path):
        #     os.remove(last_path)

        # Rename and relocate extracted file to final location
        os.rename(extracted_path, last_path)

        from time import sleep
        sleep(1)  # Optional: allow file locks to settle

    else:
        logging.debug('EdgeDriver is already installed and up to date.')

    if not os.access(last_path, os.X_OK):
        os.chmod(last_path, 0o744)

    return last_path